#include "Managers/CFrameBufferManager.hpp"
#include "Opengl/OpenGLExtensions.h"

#include "ErrorCheck.h"


//////////////////////////////////////////////////////////////////////////
//Name: Frame Buffer Object
//Desc: Holds information aboout Frame Buffer Object
//////////////////////////////////////////////////////////////////////////

CFbo::CFbo(unsigned int nSizeX,unsigned int nSizeY)
:m_nSizeX(nSizeX)
,m_nSizeY(nSizeY)
{
	
	glGenFramebuffersEXT(1, &m_nFrameBuffer);													
	glGenTextures(1, &m_nColorTex);												
	glGenRenderbuffersEXT(1, &m_nDepthBuffer);											

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, m_nFrameBuffer);								

	//Create Color Texture
	glBindTexture(GL_TEXTURE_2D, m_nColorTex);										
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);				
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);	
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE_EXT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE_EXT);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, m_nSizeX, m_nSizeY, 0,GL_RGBA, GL_INT, NULL);	
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT,GL_COLOR_ATTACHMENT0_EXT,GL_TEXTURE_2D, m_nColorTex, 0); 

	//Create Depth Buffer
	glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, m_nDepthBuffer);							
	glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT,GL_DEPTH_COMPONENT24,m_nSizeX, m_nSizeY);	
	glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT,GL_DEPTH_ATTACHMENT_EXT,GL_RENDERBUFFER_EXT, m_nDepthBuffer); 
	//Switch to default Frame Buffer
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);		


}
CFbo::~CFbo()
{
	glDeleteTextures(1,&m_nColorTex);
	glDeleteRenderbuffersEXT(1,&m_nDepthBuffer);
	glDeleteFramebuffersEXT(1,&m_nFrameBuffer);
	
}
	
	
//////////////////////////////////////////////////////////////////////////
//Name: Frame Buffer Object Manager
//Desc: Manages Objects
//////////////////////////////////////////////////////////////////////////
	
	
CFrameBufferManager	*g_pcFboMan = NULL;

CFrameBufferManager::CFrameBufferManager()
{

}
CFrameBufferManager::~CFrameBufferManager()
{
	m_cItems.ClearData();
	m_cItems.Clear();

}

CFbo*	CFrameBufferManager::CreateFrameBufferObject(const char *strName,unsigned int nSizeX,unsigned int nSizeY)
{
	CFbo*	pcResult;
	
	if(FindItem(strName,pcResult))FATALERROR("FBO with name %s already exists",strName);
	else
	{
		pcResult = new CFbo(nSizeX,nSizeY);


		CFrameBufferManager::CheckFboError();
		
		AddItem(strName,pcResult);
		
		return pcResult;
	}
	return NULL;
}
CFbo*	CFrameBufferManager::GetFrameBufferObject(const char *strName)
{
	CFbo*	pcResult;
	if(FindItem(strName,pcResult))return pcResult;
	else
	{
		FATALERROR("Failed to find FBO with name %s",strName);
		return NULL;
	}
}

//If you pass NULL, you'll switch to default FrameBuffer
void	CFrameBufferManager::SwitchToFbo(CFbo * pcFbo)
{
	if(pcFbo)
	{
		glEnable(GL_TEXTURE_2D);
		glBindTexture(GL_TEXTURE_2D, 0);
		glViewport (0, 0, pcFbo->m_nSizeX, pcFbo->m_nSizeY);	
		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, pcFbo->m_nFrameBuffer);
		CFrameBufferManager::CheckFboError();
	}
	else
	{
		glViewport(0,0,g_nWd,g_nHt);
		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
	}
}
void	CFrameBufferManager::BindFboColorTexture(CFbo * pcFbo)
{
	glBindTexture(GL_TEXTURE_2D, pcFbo->m_nColorTex);
}

void	CFrameBufferManager::IsSupported()
{
	if(!GL_EXT_framebuffer_object_supported)FATALERROR("Frame Buffer Object extension not supported!");
}
void	CFrameBufferManager::CheckFboError()
{
	CHECK_GL_ERROR();
	switch(glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT)) 
	{                                          
    case GL_FRAMEBUFFER_COMPLETE_EXT: 
		return;
		break; 
	case GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT_EXT:
		FATALERROR("GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT_EXT"); 
		break;
	case GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT_EXT:
		FATALERROR("GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT_EXT");
		break;
	case GL_FRAMEBUFFER_INCOMPLETE_DUPLICATE_ATTACHMENT_EXT:
		FATALERROR("GL_FRAMEBUFFER_INCOMPLETE_DUPLICATE_ATTACHMENT_EXT");
		break;
	case GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT:
		FATALERROR("GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT");
		break;
	case GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT:
		FATALERROR("GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT");
		break;
	case GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER_EXT:
		FATALERROR("GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER_EXT");
		break;
	case GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER_EXT:
		FATALERROR("GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER_EXT");
		break;
	case GL_FRAMEBUFFER_UNSUPPORTED_EXT: 
		FATALERROR("GL_FRAMEBUFFER_UNSUPPORTED_EXT");
		break;                                   
    default:   
		FATALERROR("Unknown Frame Buffer Object error");
	}                                                                                  

}