#include	"Managers/CTextureManager.hpp"
#include    "Opengl/OpenGLExtensions.h"
#include    "Globals/Globals.hpp"



#include	"ErrorCheck.h"

CTextureManager *g_pcTexMan=NULL;

CTextureManager::CTextureManager()
:CManager()
{

}
CTextureManager::~CTextureManager() 
{
	ReleaseTextures(); 
	ilShutDown();
}

bool CTextureManager::Init( )
{
	ReleaseTextures();

	//Initialize DevIL library
	ilInit();

	m_pcDefaultTex2D = new CTexture;
	if( !m_pcDefaultTex2D )
		return false;
		
	if( CreateDefaultTexture2D( 64 ) )
	{
		glGenTextures( 1, &m_pcDefaultTex2D->m_nTextureId );
		CreateGLTexture( m_pcDefaultTex2D );

		m_pcDefaultTex2D->m_pcTexels = NULL;

		AddItem("DefaultTexture",m_pcDefaultTex2D);

		return true;
	}

	return false;
}
void CTextureManager::ReleaseTextures( void )
{
	CTexture			 *pcCurTex;
	HashItem <CTexture*> *pcItem=m_cItems.Next(NULL);
	
	while(pcItem)
	{
		pcCurTex=pcItem->data;
		if(m_pcDefaultTex2D==pcCurTex || m_pcDefaultTex2D->m_nTextureId!=pcCurTex->m_nTextureId)
		{
			if(pcCurTex->m_nDevilID!=0)
			{
				ilDeleteImages(1,&pcCurTex->m_nDevilID);
				CHECK_DI_ERROR();
			
				glDeleteTextures( 1, &pcCurTex->m_nTextureId );
				CHECK_GL_ERROR();
			}
		}
		delete pcCurTex;
		pcItem=m_cItems.Next(pcItem);
	}
	m_cItems.Clear();
	
	m_iTextureCount = 0;
}

//////////////////////////////////////////////////////////
//Name: Some Functions
//Desc:
//////////////////////////////////////////////////////////
bool CTextureManager::CreateDefaultTexture2D( unsigned int size )
{
	if( !m_pcDefaultTex2D )
		return false;


	//The Default Image is in 0 place
	m_pcDefaultTex2D->m_nDevilID=0;
	ilBindImage(0);
	iluSwapColours();

	m_pcDefaultTex2D->m_nWidth = ilGetInteger(IL_IMAGE_WIDTH);
	m_pcDefaultTex2D->m_nHeight = ilGetInteger(IL_IMAGE_HEIGHT);
	m_pcDefaultTex2D->m_nFormat = GL_RGB;
	m_pcDefaultTex2D->m_nInternalFormat = ilGetInteger(IL_IMAGE_BYTES_PER_PIXEL) ;
	unsigned int iSize = m_pcDefaultTex2D->m_nWidth * m_pcDefaultTex2D->m_nHeight
		* m_pcDefaultTex2D->m_nInternalFormat;
	m_pcDefaultTex2D->m_pcTexels=ilGetData();

	CHECK_DI_ERROR();

	if( !m_pcDefaultTex2D->m_pcTexels )
		return false;

	return true;
}
void CTextureManager::CreateGLTexture( CTexture *img )
{
	glBindTexture( GL_TEXTURE_2D, img->m_nTextureId );

	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );


//	glTexImage2D( GL_TEXTURE_2D, 0, img->internalFormat, img->width, img->m_nHeight,
//					0, img->m_nFormat, GL_UNSIGNED_BYTE, img->m_pcTexels );
	gluBuild2DMipmaps( GL_TEXTURE_2D, img->m_nInternalFormat, img->m_nWidth, img->m_nHeight,
					img->m_nFormat, GL_UNSIGNED_BYTE, img->m_pcTexels );
}


bool CTextureManager::LoadTextureFromFile(const char *szFileName, CTexture **img, bool bFlipVert,bool bForceRGBA )
{
	
	FILE *fp=fopen(szFileName,"rb");
	if(fp==NULL)
	{
		return false;
	}
	else
	{
		fclose(fp);
		*img=new CTexture();

		ilGenImages(1,&((*img)->m_nDevilID));
		ilBindImage((*img)->m_nDevilID);
		if(ilLoadImage(const_cast<char*>(szFileName)))
		{ 
			if(bForceRGBA)ilConvertImage(IL_RGBA, IL_UNSIGNED_BYTE);
			if(bFlipVert)iluFlipImage();
			(*img)->m_nWidth = ilGetInteger(IL_IMAGE_WIDTH);
			(*img)->m_nHeight = ilGetInteger(IL_IMAGE_HEIGHT);
			(*img)->m_nDepth = ilGetInteger( IL_IMAGE_DEPTH );
			(*img)->m_nFormat = ilGetInteger(IL_IMAGE_FORMAT);
			(*img)->m_nInternalFormat = ilGetInteger(IL_IMAGE_BYTES_PER_PIXEL) ;
			(*img)->m_pcTexels=ilGetData();
			
			CHECK_DI_ERROR();
			AddItem(szFileName,*img);
			return true;
		}
		else
		{
			FATALERROR("This shouldn't happen");
			return false;
		}
	}
}
//////////////////////////////////////////////////////////
//Name: Some Functions
//Desc:
//////////////////////////////////////////////////////////
CTexture*	 CTextureManager::CreateTexture(const char *szFileName,int nWidth,int nLength,int channels,int type)
{
	GetFullPath(g_sApp.strMainDir,szFileName,m_strFullPath);
	
	CTexture *img = new CTexture();
	img->m_nWidth=nWidth;
	img->m_nHeight=nLength;
	img->m_nInternalFormat=channels;
	img->m_nFormat=type;


	// Allocate and init memory for the image array and point to it from pTexture
	img->m_pcTexels = new GLubyte [nWidth * nLength * channels];
	memset(img->m_pcTexels, 0,nWidth * nLength * channels * sizeof(GLubyte));	

	// Register the texture with OpenGL and bind it to the texture ID
	glGenTextures(1, &img->m_nTextureId);								
	glBindTexture(GL_TEXTURE_2D, img->m_nTextureId);					

	// Create the texture and store it on the video card
	glTexImage2D(GL_TEXTURE_2D, 0, channels, nWidth, nLength, 0, type, GL_UNSIGNED_BYTE, img->m_pcTexels);						

	// Set the texture quality
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);

	// Since we stored the texture space with OpenGL, we can delete the image data
	delete [] img->m_pcTexels;	
	img->m_pcTexels = NULL;

	AddItem(m_strFullPath,img);

	m_iTextureCount++;

	return img;

}
CTexture*    CTextureManager::LoadTexture2D(const char *szFileName, bool bFlipVert /* = false */ ,bool bForceRGBA)
{
	GetFullPath(g_sApp.strMainDir,szFileName,m_strFullPath);

	CTexture *img = GetTexture( szFileName );

	if( IsDefaultTexture( img ) )
	{
		if( LoadTextureFromFile( m_strFullPath, &img, bFlipVert, bForceRGBA) )
		{
			glGenTextures( 1, &img->m_nTextureId );
			CreateGLTexture( img );
			img->m_pcTexels		= NULL;
			img->m_eTextureType	= CTexture::TT_Tex2D;
			m_iTextureCount++;
		}
		else
		{
			DEBUGMSG("CTextureManager : failed to load %s ! Using default tex (id=%d)",
                m_strFullPath, m_pcDefaultTex2D->m_nTextureId);
			img = m_pcDefaultTex2D;
		}
	}

	return img;
}

CTexture*	 CTextureManager::LoadNoiseTexture(const char *szFileName, bool bFlipVert)
{
	
	CTexture *img = GetTexture( szFileName );
	if( IsDefaultTexture( img ) )
	{
		GetFullPath(g_sApp.strMainDir,szFileName,m_strFullPath);
		if( LoadTextureFromFile( m_strFullPath, &img, bFlipVert ) )
		{

			ASSERT(img->m_nDepth>1,"Image depth is incorrect:%d",img->m_nDepth);
			// request 1 texture name from OpenGL
			glGenTextures(1, &img->m_nTextureId);	
			// tell OpenGL we're going to be setting up the texture name it gave us	
			glBindTexture(GL_TEXTURE_3D, img->m_nTextureId);	
			// when this texture needs to be shrunk to fit on small polygons, use linear interpolation of the texels to determine the color
			glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
			// when this texture needs to be magnified to fit on a big polygon, use linear interpolation of the texels to determine the color
			glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
			// we want the texture to repeat over the S axis, so if we specify coordinates out of range we still get textured.
			glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_S, GL_REPEAT);
			// same as above for T axis
			glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_T, GL_REPEAT);
			// same as above for R axis
			glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_EDGE);

			glTexImage3D(GL_TEXTURE_3D, 0,img->m_nInternalFormat, img->m_nWidth, img->m_nHeight, img->m_nDepth, 0, img->m_nFormat, GL_UNSIGNED_BYTE, img->m_pcTexels);
	
			img->m_pcTexels = NULL;
			img->m_eTextureType	= CTexture::TT_Noise;

			m_iTextureCount++;
		}
		else
		{
			DEBUGMSG("CTextureManager : failed to load %s ! Using default tex (id=%d)",
				m_strFullPath, m_pcDefaultTex2D->m_nTextureId);

			img = m_pcDefaultTex2D;
		}
	}
	return img;

}

CTexture*	 CTextureManager::LoadTexture3D(int nWidth, int nLength, int nHeight,CTexName *pcFileName)
{
	int i,j;

	int		  nBitsPerPixel,nOneTexSize,nFullSize,nOffset;

	CTexture **pcTex2D;
	CTexture *pcTex3D=new CTexture;

	pcTex2D=new CTexture*[nHeight];

	for(i=0;i<nHeight;i++)
	{
		pcTex2D[i]=NULL;
	}

	nBitsPerPixel=3;
	nOneTexSize=nWidth*nLength*nBitsPerPixel;
	nFullSize=nOneTexSize*nHeight;
	bool bLoadSuccesful=true;


	for(i=0;i<nHeight;i++)
	{

		GetFullPath(g_sApp.strMainDir,pcFileName[i].strName,m_strFullPath);
		if(!LoadTextureFromFile(m_strFullPath,&pcTex2D[i],false))
		{

			DEBUGMSG("CTextureManager : failed to load %s ! Using default tex (id=%d)",
				pcFileName[i].strName, -1);// pcTex2D[i]->m_nTextureId);
		}
		else
		{
			if(pcTex2D[i]->m_nHeight!=nLength || pcTex2D[i]->m_nWidth!=nWidth)
			{
				DEBUGMSG("Texture Size doesnt Match\nTexName:%s Height:%d=?=%d Width:%d=?=%d",
					pcFileName[i].strName,pcTex2D[i]->m_nHeight,nLength,pcTex2D[i]->m_nWidth,nWidth);
				bLoadSuccesful=false;
			}
		}

	}

	if(bLoadSuccesful)
	{
		pcTex3D->m_pcTexels=new GLubyte[nFullSize];
		memset(pcTex3D->m_pcTexels,128,nFullSize);

		for(j=0;j<nHeight;j++)
		{
			nOffset=nOneTexSize*j;
			for(i=0;i<nOneTexSize;i++)
			{
				pcTex3D->m_pcTexels[i+nOffset]=pcTex2D[j]->m_pcTexels[i];
			}
		}
	




		// request 1 texture name from OpenGL
		glGenTextures(1, &pcTex3D->m_nTextureId);	
		// tell OpenGL we're going to be setting up the texture name it gave us	
		glBindTexture(GL_TEXTURE_3D, pcTex3D->m_nTextureId);	
		// when this texture needs to be shrunk to fit on small polygons, use linear interpolation of the texels to determine the color
		glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		// when this texture needs to be magnified to fit on a big polygon, use linear interpolation of the texels to determine the color
		glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		// we want the texture to repeat over the S axis, so if we specify coordinates out of range we still get textured.
		glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		// same as above for T axis
		glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_T, GL_REPEAT);
		// same as above for R axis
		glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_EDGE);
		// this is a 3d texture, level 0 (max detail), GL should store it in RGB8 format, its WIDTHxHEIGHTxDEPTH in size, 
		// it doesnt have a border, we're giving it to GL in RGB format as a series of unsigned bytes, and texels is where the texel data is.
		glTexImage3D(GL_TEXTURE_3D, 0, GL_RGB8, nWidth, nLength, nHeight, 0, GL_RGB, GL_UNSIGNED_BYTE, pcTex3D->m_pcTexels);


		delete [] pcTex3D->m_pcTexels;
		pcTex3D->m_pcTexels = NULL;
		pcTex3D->m_eTextureType = CTexture::TT_Tex3D;

	
		
		//AddItem("Texture3D",pcTex3D);
		
		m_iTextureCount++;

	}
	


	for(i=0;i<nHeight;i++)
	{
		if(pcTex2D[i]!=NULL)
			delete pcTex2D[i];
	}
	delete []pcTex2D;

	if(bLoadSuccesful)return pcTex3D;
	else return 0;

}


CTexture*	 CTextureManager::LoadCubeMap  (const char *szFileName,
											const char *srtExtension )
{
	static const unsigned _nCubeMapCnt = 6;
	static const char *_strEndLetters[_nCubeMapCnt] = {"_posx","_negx","_posy","_negy","_posz","_negz"};

	static int		   _iCubeMapType [_nCubeMapCnt] =
	{
		GL_TEXTURE_CUBE_MAP_POSITIVE_X,
		GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
		GL_TEXTURE_CUBE_MAP_POSITIVE_Y,
		GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
		GL_TEXTURE_CUBE_MAP_POSITIVE_Z,
		GL_TEXTURE_CUBE_MAP_NEGATIVE_Z
	};

	char		buf[255];
	CTexture   *pcTex2D[_nCubeMapCnt];


	CTexture *img = GetTexture( szFileName );

	if( IsDefaultTexture( img ) )
	{
		img = new CTexture();



		//glEnable        ( GL_TEXTURE_CUBE_MAP );

		glGenTextures   ( 1, &img->m_nTextureId );
		glPixelStorei	( GL_UNPACK_ALIGNMENT, 1 );
		glBindTexture   ( GL_TEXTURE_CUBE_MAP, img->m_nTextureId  );
		glTexEnvi		( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE );
		glTexParameteri ( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
		glTexParameteri ( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
		glTexParameteri ( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_EDGE );

		glTexParameteri ( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR );
		glTexParameteri ( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
		glTexParameteri ( GL_TEXTURE_CUBE_MAP, GL_GENERATE_MIPMAP_SGIS, GL_TRUE );

		for(int i=0;i<_nCubeMapCnt;i++)
		{
			sprintf(buf,"%s%s%s",szFileName,_strEndLetters[i],srtExtension);
			GetFullPath(g_sApp.strMainDir,buf,m_strFullPath);
			if(!LoadTextureFromFile( m_strFullPath, &pcTex2D[i], false ) )
			{
				FATALERROR("Failed to load %s",m_strFullPath);
			}

			glTexImage2D(_iCubeMapType[i], 0, pcTex2D[i]->m_nInternalFormat, pcTex2D[i]->m_nWidth, pcTex2D[i]->m_nHeight,
				0,pcTex2D[i]->m_nFormat, GL_UNSIGNED_BYTE, pcTex2D[i]->m_pcTexels);	
			//Devil should free memory, but i am not sure
			/*delete[](pcTex2D[i]->m_pcTexels);*/pcTex2D[i]->m_pcTexels = NULL;
		}
		img->m_eTextureType = CTexture::TT_CubeMap;

		AddItem(m_strFullPath,img);
		m_iTextureCount++;
		//glDisable	  (GL_TEXTURE_CUBE_MAP);
	}
	return img;
}



CTexture*	 CTextureManager::GetTexture(const char *szFileName )
{
	CTexture *pcTex=NULL;
	
	GetFullPath(g_sApp.strMainDir,szFileName,m_strFullPath);
	
	if(FindItem(m_strFullPath,pcTex))
	{
		return pcTex;
	}
	else
	{
		return m_pcDefaultTex2D;
	}
}
