#include "Materials/CTctMat.hpp"
#include "Materials/CTctAttributes.hpp"
#include "Managers/CShaderManager.hpp"
#include "Managers/CTextureManager.hpp"

#include "Globals/Globals.hpp"
#include "Camera/Camera.h"
#include "Utils/MiscMacros.hpp"
#include "Utils/CFileReader.hpp"
#include "TinyXML/tinyxml.h"
#include "Props/CAttribute.hpp"


#include "ErrorCheck.h"


/////////////////////////////////////////////////////////
//Name: CTctMap
//Desc: Simple Map written specially for 3dsMax
/////////////////////////////////////////////////////////

CTctMap::CTctMap(const char *strName )
{
	m_pcTexture		= NULL;
	m_strMapName	= NULL;
	m_fMapAmount	= 1.0f;
	m_vUVOffset		= Vector2D(0.0f,0.0f);
	m_vUVTiling		= Vector2D(1.0f,1.0f);
	m_fUVAngle		= 0.0f;
	m_fUVBlur		= 1.0f;
	m_fUVBlurOffset	= 0.0f;
	m_fNoiseAmt		= 1.0f;
	m_fNoiseSize	= 1.0f;
	m_nNoiseLvl		= 1;
	m_fNoisePhase	= 0.0f;
	m_bEnabled		= false;
	m_nTextureLevel = 0;
	SetMapName(strName);
}
CTctMap::~CTctMap()
{
	SAFE_FREE(m_strMapName);
}

void CTctMap::SetMapName(const char *strName)
{
	strAllocNCopy(&m_strMapName,strName);
}


/////////////////////////////////////////////////////////
//Name: CTctMat
//Desc: Complexed material for TCT model
/////////////////////////////////////////////////////////

const GLenum CTctMat::_eTexTargets[4] =
{
	GL_TEXTURE_2D,
	GL_TEXTURE_3D,
	GL_TEXTURE_3D,
	GL_TEXTURE_CUBE_MAP
};

CTctMat::CTctMat(const char *strFileName):CMaterial(strFileName)
{
	m_vAmbient		= Vector3D(1.0f,1.0f,1.0f);
	m_vDiffuse		= Vector3D(1.0f,1.0f,1.0f);
	m_vSpecular		= Vector3D(1.0f,1.0f,1.0f);
	
	m_fShine		= 0.1f;
	m_fShineStr		= 0.0f;
	m_fTransparency = 0.0f;
	m_fWireSize		= 1.0f;
	
	m_nMaterialID	= 0;


	CFileReader cFile;
	if(cFile.OpenFile(strFileName))
	{
		TiXmlDocument  xcDoc("Material");
		TiXmlNode	  *xcRoot;
		TiXmlElement  *xcMaps;
		TiXmlElement  *xcShaders;
		TiXmlElement  *xcName;
		TiXmlElement  *xcCurrentMap,*xcCurrentAttr;
		xcDoc.Parse((char*)cFile.GetBuffer());
		xcRoot = xcDoc.FirstChildElement();
		
		xcName	  = xcRoot->FirstChildElement("Name")->ToElement();
		xcMaps	  = xcRoot->FirstChildElement("Maps")->ToElement();
		xcShaders = xcRoot->FirstChildElement("Shader")!=NULL?
			xcRoot->FirstChildElement("Shader")->ToElement():
			NULL;
		ASSERT_EXP(xcName!=NULL && xcMaps!=NULL);

		//Read Shaders;
		if(xcShaders)
		{
			m_pcShader = g_pcShaderMan->LoadFromFile(
				xcShaders->Attribute("Vertex"),
				xcShaders->Attribute("Fragment"));

		}
		else
		{
			m_pcShader = g_pcShaderMan->LoadFromFile(
				"Basic.vert",
				"Basic.frag");
		}

		

		//Read Maps
		for( xcCurrentAttr = xcRoot->FirstChildElement();
			 xcCurrentAttr;
			 xcCurrentAttr = xcCurrentAttr->NextSiblingElement())
		{
			if(xcCurrentAttr==xcName || xcCurrentAttr==xcShaders || xcCurrentAttr==xcMaps)continue;
			if(m_pcShader->CheckUniformLocation(xcCurrentAttr->Value()))
			{
				ProcessAttribute(xcCurrentAttr->Value(),xcCurrentAttr->FirstAttribute());
			}
			else
			{
				DEBUGMSG("Failed to find attribute %s in shader \n %s \n Material File :%s",
					xcCurrentAttr->Value(),m_pcShader->GetName(),strFileName);
			}
		}

		//Read Maps
		for( xcCurrentMap = xcMaps->FirstChildElement();
			 xcCurrentMap;
			 xcCurrentMap = xcCurrentMap->NextSiblingElement())
		{
			if(stricmp(xcCurrentMap->Value(),"BumpMap")==0)continue;
			CTctMap *pcNewMap = new CTctMap(xcCurrentMap->Value());
			ASSERT_EXP(xcCurrentMap->Attribute("sName"));

			const char *strTexType = xcCurrentMap->Attribute("sType");
			if(strTexType==NULL || stricmp(strTexType,"Texture2D")==0)
				pcNewMap->SetTexture(g_pcTexMan->LoadTexture2D(xcCurrentMap->Attribute("sName"),true));
			else if(stricmp(strTexType,"Noise")==0)
				pcNewMap->SetTexture(g_pcTexMan->LoadNoiseTexture(xcCurrentMap->Attribute("sName")));
			else if(stricmp(strTexType,"CubeMap")==0)
				pcNewMap->SetTexture(g_pcTexMan->LoadCubeMap(xcCurrentMap->Attribute("sName"),".png"));//FIX EXTENSION
			else FATALERROR("Unknown Texture Type %s",xcCurrentMap->Attribute("sName"));

		
				
			m_cMaps.AddAtEnd(pcNewMap);
		}

		cFile.CloseFile();
		
	}
	else  DEBUGMSG("Failed to open %s",strFileName);

}
CTctMat::~CTctMat()
{
	m_cMaps.ClearData();
	m_cMaps.Clear();
	m_cAttrs.ClearData();
	m_cAttrs.Clear();
}


void  CTctMat::ProcessAttribute(const char *strName,TiXmlAttribute *pcAttribute)
{
	const char * strXmlName = pcAttribute->Name();

	CTctAttribute *pcNewAttribute = NULL;
	CAttribute	   cAttr;
	cAttr.SetString(pcAttribute->Value());

	switch(strXmlName[0])
	{
	case 's':
		//Do nothing
		break;
	case 'f':
		if(pcAttribute->Value()[0]=='$')
		{
			if(strcmp(&(pcAttribute->Value()[1]),"Time")==0)pcNewAttribute = new CTctAttrFloatPtr(strName,&ShaderVals::g_fShTime);
			else if(strcmp(&(pcAttribute->Value()[1]),"LoopTime")==0)pcNewAttribute = new CTctAttrFloatPtr(strName,&ShaderVals::g_fShLoopTime);
			else if(strcmp(&(pcAttribute->Value()[1]),"HillTime")==0)pcNewAttribute = new CTctAttrFloatPtr(strName,&ShaderVals::g_fShHillTime);
			else if(strcmp(&(pcAttribute->Value()[1]),"Sinus")==0)pcNewAttribute = new CTctAttrFloatPtr(strName,&ShaderVals::g_fShSinus);
			else if(strcmp(&(pcAttribute->Value()[1]),"Cosinus")==0)pcNewAttribute = new CTctAttrFloatPtr(strName,&ShaderVals::g_fShCosinus);
			else DEBUGMSG("Unkown dynamic variable %s",pcAttribute->Value());
			
		}
		else
		{
			pcNewAttribute = new CTctAttrFloat(strName,cAttr.ConvertToFloat());
		}
		break;
	case 'i':
		pcNewAttribute = new CTctAttrInt(strName,cAttr.ConvertToInt());
		break;
	case 'd':
		pcNewAttribute = new CTctAttrInt(strName,cAttr.ConvertToInt());
		break;
	case 'b':
		//Do nothing
		break;
	case 'v':
		if(pcAttribute->Value()[0]=='$')
		{
			const char *s =&(pcAttribute->Value()[1]);
			if(strcmp(&(pcAttribute->Value()[1]),"CameraEye")==0)pcNewAttribute = new CTctAttrVec3Ptr(strName,&ShaderVals::g_vEye);
			else if(strcmp(&(pcAttribute->Value()[1]),"CurPosition")==0)pcNewAttribute = new CTctAttrVec3Ptr(strName,&ShaderVals::g_vCurPosition);
			else if(strcmp(&(pcAttribute->Value()[1]),"CurScale")==0)pcNewAttribute = new CTctAttrVec3Ptr(strName,&ShaderVals::g_vCurScale);
		
			else DEBUGMSG("Unkown dynamic variable %s",pcAttribute->Value());
		}
		else
		{
			if(strXmlName[1]=='2')pcNewAttribute = new CTctAttrVec2(strName,cAttr.ConvertToVector2D());
			else if(strXmlName[1]=='3')pcNewAttribute = new CTctAttrVec3(strName,cAttr.ConvertToVector3D());
			else if(strXmlName[1]=='4')pcNewAttribute = new CTctAttrVec4(strName,cAttr.ConvertToVector4D());
			else DEBUGMSG("Unknown in attribute:%s Element %s",strXmlName,strName);
		}
		break;
	case 'm':
		/*if(strName[1]=='3')pcTarget->ConvertToMatrix3D();
		else if(strName[1]=='4')pcTarget->ConvertToMatrix4D();
		else DEBUGMSG("Unknown matrix in attribute:%s",pcTarget->GetName());
		*/
		break;
	default:
		DEBUGMSG("Unknown Type in attribute:%s Element %s",strXmlName,strName);
	}
	if(pcNewAttribute)m_cAttrs.Add(pcNewAttribute);
	

}
void			CTctMat::BeginUse()
{


	if(m_pcShader->IfShaderOperationsLocked()==false)
	{
		if(m_pcShader)
		{
			m_pcShader->begin();
			//Set Maps
			int nMapCnt = 0;
			for(LinkedListItem<CTctMap*>*pItem = m_cMaps.Next(NULL);
				pItem;
				pItem = m_cMaps.Next(pItem))
			{
				glActiveTextureARB(GL_TEXTURE0_ARB+nMapCnt);
				ASSERT_EXP(pItem->data->GetTexture()->GetTexType()<4);
				glEnable(_eTexTargets[pItem->data->GetTexture()->GetTexType()]);
				glBindTexture(_eTexTargets[pItem->data->GetTexture()->GetTexType()],pItem->data->GetTexture()->GetTextureId());

				CHECK_GL_ERROR();

				m_pcShader->sendUniform1i(pItem->data->GetMapName(),nMapCnt);
				nMapCnt++;
			}

			//Set Attributes
			for(LinkedListItem<CTctAttribute*>*pItem = m_cAttrs.Next(NULL);
				pItem;
				pItem = m_cAttrs.Next(pItem))
			{
				pItem->data->SetAttribute(m_pcShader);
			}

		}
		else
		{
			if(m_cMaps.Next(NULL))
			{
				glPushAttrib(GL_ENABLE_BIT);
				glDisable(GL_CULL_FACE);
				glEnable(GL_ALPHA_TEST);
				glAlphaFunc(GL_GREATER,0.9f);
				
				glActiveTextureARB(GL_TEXTURE0_ARB);
				glEnable(GL_TEXTURE_2D);
				glBindTexture(GL_TEXTURE_2D,m_cMaps.Next(NULL)->data->GetTexture()->GetTextureId());
			}
		}
	}
}

void			CTctMat::EndUse()
{
	if(m_pcShader->IfShaderOperationsLocked()==false)
	{
		if(m_pcShader)
		{
			m_pcShader->end();
			int nMapCnt = 0;
			for(LinkedListItem<CTctMap*>*pItem = m_cMaps.Next(m_cMaps.Next(NULL));
				pItem;
				pItem = m_cMaps.Next(pItem))
			{
				glActiveTextureARB(GL_TEXTURE0_ARB+nMapCnt);
				glDisable(_eTexTargets[pItem->data->GetTexture()->GetTexType()]);
				CHECK_GL_ERROR();
				nMapCnt++;
			}
			glActiveTextureARB(GL_TEXTURE0_ARB);
			
			//This shouldnt be here, but there's an error somewhere
			glDisable(GL_TEXTURE_CUBE_MAP);
		}
		else
		{
			if(m_cMaps.Next(NULL))glPopAttrib();
		}
	
	}
}

CShaderObject * CTctMat::GetShader()
{
	return NULL;
}

