#include "Utils/CVBOBuffer.h"
#include "Globals/Globals.hpp"
#include "Utils/MiscMacros.hpp"

#include "ErrorCheck.h"


CVBOBuffer::CVBOBuffer(unsigned _nMaxElements,int _sVBODataType,GLenum _nDrawingMode,
					   bool _bUsingIndices,unsigned _nIndexCount)
:m_nDrawingMode(_nDrawingMode)
,m_nMaxElements(_nMaxElements)
,m_sVBODataType(_sVBODataType)
,m_bUsingIndices(_bUsingIndices)
,m_nMaxIndices(_nIndexCount)
{

	
	if(!IsVBOSupported())
		FATALERROR("VBO isnt supported");
	else
	{
	
		SetValuesToZero();
		
		if((m_sVBODataType & VSD_TEX0) && (m_sVBODataType & VSD_TEX3D))
			FATALERROR("VSD_TEX0 and VSD_TEX3D are enabled, only one can be enabled");

		if(m_bUsingIndices)
		{
			m_pcIndices = new unsigned int[m_nMaxIndices];
		}
		if(m_sVBODataType & VSD_VERTEX)
		{
			m_pcVertices		= new Vector3D[m_nMaxElements];
		}
		else FATALERROR("VBO Buffer without vertices?!");
		
		if(m_sVBODataType & VSD_COLORS)
		{
			m_pcColors			= new Vector4D[m_nMaxElements];
		}
		if(m_sVBODataType & VSD_TEX0)
		{
			m_pcTexCoords0		= new Vector2D[m_nMaxElements];
		}
		if(m_sVBODataType & VSD_TEX1)
		{
			m_pcTexCoords1		= new Vector2D[m_nMaxElements];
		}
		if(m_sVBODataType & VSD_NORMALS)
		{
			m_pcNormals			= new Vector3D[m_nMaxElements];
		}
		if(m_sVBODataType & VSD_TEX3D)
		{
			m_pcTexCoords3D0	= new Vector3D[m_nMaxElements];
		}
	}

}
CVBOBuffer::~CVBOBuffer()
{
	Clear();
}
void CVBOBuffer::SetValuesToZero()
{
	int i;
	for(i=0;i<6;i++)
		m_nDataStreamType[i]=GL_STATIC_DRAW_ARB;

	m_pcVertices	= NULL;	
	m_pcColors		= NULL;
	m_pcTexCoords0	= NULL;
	m_pcTexCoords1	= NULL;							
	m_pcNormals		= NULL;				
	m_pcTexCoords3D0= NULL;
	m_pcIndices		= NULL;


	

	m_nVBOVertices		= 0;	
	m_nVBOColors		= 0;
	m_nVBOTexCoords0	= 0;					
	m_nVBOTexCoords1	= 0;
	m_nVBONormals		= 0;
	m_nVBOTexCoords3D0  = 0;
	m_nVBOIndices		= 0;

	m_nVerticesCnt		= 0;								
	m_nColorsCnt		= 0;
	m_nTexCoords0Cnt	= 0;							
	m_nTexCoords1Cnt	= 0;
	m_nNormalsCnt		= 0;	
	m_nTexCoords3D0Cnt  = 0;
	m_nIndexCnt			= 0;

}
bool CVBOBuffer::IsVBOSupported()
{
	return GL_ARB_vertex_buffer_object_supported;
}
void CVBOBuffer::SetDataStreamType(VBO_STOREDATA _nStoreType,GLenum _eDataType)
{
	unsigned int _nDataIndex=unsigned int(log(float(_nStoreType))/log(2.0f));
	ASSERT(_nDataIndex<6,"Incorrect Data Index");
	m_nDataStreamType[_nDataIndex]=_eDataType;
}
bool CVBOBuffer::IsVBOBuilt()
{
	return 	(m_nVBOVertices		!= 0);
}

bool CVBOBuffer::ValidateIndices()
{
	if((m_sVBODataType & VSD_COLORS) && m_nVerticesCnt!=m_nColorsCnt)
		FATALERROR("Validation Failed - m_nVerticesCnt:%d m_nColorsCnt:%d",
			m_nVerticesCnt,m_nColorsCnt);
	if((m_sVBODataType & VSD_TEX0) && m_nVerticesCnt!=m_nTexCoords0Cnt)
		FATALERROR("Validation Failed - m_nVerticesCnt:%d m_nTexCoords0Cnt:%d",
			m_nVerticesCnt,m_nTexCoords0Cnt);	
	if((m_sVBODataType & VSD_TEX1) && m_nVerticesCnt!=m_nTexCoords1Cnt)
		FATALERROR("Validation Failed - m_nVerticesCnt:%d m_nTexCoords1Cnt:%d",
			m_nVerticesCnt,m_nTexCoords1Cnt);
	if((m_sVBODataType & VSD_NORMALS) && m_nVerticesCnt!=m_nNormalsCnt)
		FATALERROR("Validation Failed - m_nVerticesCnt:%d m_nNormalsCnt:%d",
			m_nVerticesCnt,m_nNormalsCnt);
	if((m_sVBODataType & VSD_TEX3D) && m_nVerticesCnt!=m_nTexCoords3D0Cnt)
		FATALERROR("Validation Failed - m_nVerticesCnt:%d m_nTexCoords3D0Cnt:%d",
			m_nVerticesCnt,m_nTexCoords3D0Cnt);
	return true;
}
void CVBOBuffer::Clear()
{


	if(m_nVBOVertices	!= 0)glDeleteBuffersARB( 1,&m_nVBOVertices );	
	if(m_nVBOColors		!= 0)glDeleteBuffersARB( 1,&m_nVBOColors );	
	if(m_nVBOTexCoords0	!= 0)glDeleteBuffersARB( 1,&m_nVBOTexCoords0 );	
	if(m_nVBOTexCoords1	!= 0)glDeleteBuffersARB( 1,&m_nVBOTexCoords1 );	
	if(m_nVBONormals	!= 0)glDeleteBuffersARB( 1,&m_nVBONormals );	
	if(m_nVBOTexCoords3D0!= 0)glDeleteBuffersARB( 1,&m_nVBOTexCoords3D0 );	
	if(m_nVBOIndices!= 0)	 glDeleteBuffersARB( 1,&m_nVBOIndices );	
	
	for(LinkedListItem<CShaderAttr*> *pcItem=m_cAttrList.Next(NULL);
		pcItem!=NULL;
		pcItem=m_cAttrList.Next(pcItem))
	{
		delete pcItem->data;
	}
	m_cAttrList.Clear();


	SAFE_DELETE_ARRAY(m_pcVertices);
	SAFE_DELETE_ARRAY(m_pcColors);
	SAFE_DELETE_ARRAY(m_pcTexCoords0);
	SAFE_DELETE_ARRAY(m_pcTexCoords1);
	SAFE_DELETE_ARRAY(m_pcNormals);
	SAFE_DELETE_ARRAY(m_pcTexCoords3D0);
	SAFE_DELETE_ARRAY(m_pcIndices);

	SetValuesToZero();

}
void CVBOBuffer::AddShaderAttribute(CShaderAttr *pcShaderAttr)
{
	ASSERT(m_nVerticesCnt==pcShaderAttr->GetCount(),
		"Vertex Count:%d isnt equal to ShaderAttr Count:%d",m_nVerticesCnt,pcShaderAttr->GetCount());
	m_cAttrList.Add(pcShaderAttr);
}

void CVBOBuffer::AddIndex(unsigned _nIndex)
{
	ASSERT(_nIndex<m_nVerticesCnt,"Index(%d) not in range[0;%d]",_nIndex,m_nVerticesCnt);
	ASSERT(m_nIndexCnt<m_nMaxIndices,"Too much indices");
	ASSERT(m_pcIndices!=NULL,"Indices not inited");
	ASSERT(m_bUsingIndices,"Not using indices");
	
	m_pcIndices[m_nIndexCnt]=_nIndex;
	m_nIndexCnt++;
}
//Vertex
void CVBOBuffer::AddVertex(Vector3D v)
{
	ASSERT(m_nVerticesCnt<m_nMaxElements,"Too much elements");
	ASSERT(m_pcVertices!=NULL,"Vertices not inited");
	m_pcVertices[m_nVerticesCnt]=v;
	m_nVerticesCnt++;
}
//Color
void CVBOBuffer::AddColor(Vector4D c)
{
	ASSERT(m_nColorsCnt<m_nMaxElements,"Too much elements");
	ASSERT(m_pcColors!=NULL,"Color not inited");
	m_pcColors[m_nColorsCnt]=c;
	m_nColorsCnt++;
}
//TexCoord0
void CVBOBuffer::AddTexCoord0(Vector2D t)
{
	ASSERT(m_nTexCoords0Cnt<m_nMaxElements,"Too much elements");
	ASSERT(m_pcTexCoords0!=NULL,"Tex0 not inited");
	m_pcTexCoords0[m_nTexCoords0Cnt]=t;
	m_nTexCoords0Cnt++;

}
//TexCoord1
void CVBOBuffer::AddTexCoord1(Vector2D t)
{
	ASSERT(m_nTexCoords1Cnt<m_nMaxElements,"Too much elements");
	ASSERT(m_pcTexCoords1!=NULL,"Tex1 not inited");
	m_pcTexCoords1[m_nTexCoords1Cnt]=t;
	m_nTexCoords1Cnt++;
}
//Normal
void CVBOBuffer::AddNormal(Vector3D n)
{
	ASSERT(m_nNormalsCnt<m_nMaxElements,"Too much elements");
	ASSERT(m_pcNormals!=NULL,"Normals not inited");
	m_pcNormals[m_nNormalsCnt]=n;
	m_nNormalsCnt++;
}
void CVBOBuffer::AddTexCoord3D0(Vector3D t)
{
	ASSERT(m_nTexCoords3D0Cnt<m_nMaxElements,"Too much elements");
	ASSERT(m_pcTexCoords3D0!=NULL,"Tex3D0 not inited");
	m_pcTexCoords3D0[m_nTexCoords3D0Cnt]=t;
	m_nTexCoords3D0Cnt++;
}
	//For Dynamic Buffers only
void	CVBOBuffer::UpdateVertexBuffer()
{
	ASSERT(m_nVBOVertices!=0,"Vertex Buffer isnt bind");
	ASSERT(m_nDataStreamType[0]==GL_DYNAMIC_DRAW_ARB,"Data Stream must be GL_DYNAMIC_DRAW_ARB");
	tagUpdateVBO(m_pcVertices	,m_nVBOVertices		,m_nVerticesCnt		,3*sizeof(float));

}
void	CVBOBuffer::UpdateNormalsBuffer()
{
	ASSERT(m_nVBONormals!=0,"Normals Buffer isnt bind");
	ASSERT(m_nDataStreamType[4]==GL_DYNAMIC_DRAW_ARB,"Data Stream must be GL_DYNAMIC_DRAW_ARB");
	tagUpdateVBO(m_pcNormals	,m_nVBONormals		,m_nNormalsCnt		,3*sizeof(float));

}
void	CVBOBuffer::UpdateColorsBuffer()
{
	ASSERT(m_nVBOColors!=0,"Color Buffer isnt bind");
	ASSERT(m_nDataStreamType[1]==GL_DYNAMIC_DRAW_ARB,"Data Stream must be GL_DYNAMIC_DRAW_ARB");
	tagUpdateVBO(m_pcColors	,m_nVBOColors		,m_nColorsCnt, 4*sizeof(float));

}

void CVBOBuffer::BuildVBO()
{
	tagBuildVBO(m_pcVertices	,m_nVBOVertices		,m_nVerticesCnt		,3*sizeof(float),m_nDataStreamType[0]);
	tagBuildVBO(m_pcColors		,m_nVBOColors		,m_nColorsCnt		,4*sizeof(float),m_nDataStreamType[1]);
	tagBuildVBO(m_pcTexCoords0	,m_nVBOTexCoords0	,m_nTexCoords0Cnt	,2*sizeof(float),m_nDataStreamType[2]);
	tagBuildVBO(m_pcTexCoords1	,m_nVBOTexCoords1	,m_nTexCoords1Cnt	,2*sizeof(float),m_nDataStreamType[3]);
	tagBuildVBO(m_pcNormals		,m_nVBONormals		,m_nNormalsCnt		,3*sizeof(float),m_nDataStreamType[4]);
	tagBuildVBO(m_pcTexCoords3D0,m_nVBOTexCoords3D0	,m_nTexCoords3D0Cnt	,3*sizeof(float),m_nDataStreamType[5]);
	
	
	if(m_bUsingIndices && ValidateIndices())
	{
		ASSERT(m_nVBOIndices==0,"Indices Buffer alreay built!!!");
		glGenBuffersARB(1, &m_nVBOIndices);
		glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB, m_nVBOIndices);
	    glBufferDataARB(GL_ELEMENT_ARRAY_BUFFER_ARB, m_nIndexCnt*sizeof(unsigned int), m_pcIndices, GL_STATIC_DRAW_ARB);
	}

	


}


void	CVBOBuffer::BeginRender()
{
	if(m_sVBODataType & VSD_VERTEX)
	{
		glEnableClientState(GL_VERTEX_ARRAY); 
		glBindBufferARB( GL_ARRAY_BUFFER_ARB, GetVertices() );
		glVertexPointer(3, GL_FLOAT, 0,(char *) NULL);
	}
	if(m_sVBODataType & VSD_COLORS)
	{
		glEnableClientState(GL_COLOR_ARRAY); 
		glBindBufferARB( GL_ARRAY_BUFFER_ARB, GetColors() );
		glColorPointer(4, GL_FLOAT, 0,(char *) NULL);
	}
	if(m_sVBODataType & VSD_TEX0)
	{
		glClientActiveTextureARB(GL_TEXTURE0_ARB);
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glBindBufferARB( GL_ARRAY_BUFFER_ARB, GetTexCoord0() );
		glTexCoordPointer(2, GL_FLOAT,0,(char *) NULL);

	}
	if(m_sVBODataType & VSD_TEX1)
	{
		glClientActiveTextureARB(GL_TEXTURE1_ARB);
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glBindBufferARB( GL_ARRAY_BUFFER_ARB, GetTexCoord1() );
		glTexCoordPointer(2, GL_FLOAT,0,(char *) NULL);

	}
	if(m_sVBODataType & VSD_NORMALS)
	{
		glEnableClientState(GL_NORMAL_ARRAY);
		glBindBufferARB( GL_ARRAY_BUFFER_ARB, GetNormals());
		glNormalPointer( GL_FLOAT, 0,(char *) NULL);
	}
	if(m_sVBODataType & VSD_TEX3D)
	{
		glClientActiveTextureARB(GL_TEXTURE0_ARB);
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glBindBufferARB( GL_ARRAY_BUFFER_ARB, GetTexCoord3D0() );
		glTexCoordPointer(3, GL_FLOAT,0,(char *) NULL);
	}
	
	for(LinkedListItem<CShaderAttr*> *pcItem=m_cAttrList.Next(NULL);
		pcItem!=NULL;
		pcItem=m_cAttrList.Next(pcItem))
	{
		pcItem->data->BeginUse();
	}


	if(m_bUsingIndices)
	{
		glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB, m_nVBOIndices);
	}


}
void	CVBOBuffer::Render()
{
#ifdef CALC_REND_TRIS
	g_nVertRendered+=GetVertexCount();
#endif
	if(m_bUsingIndices)
	{
		glDrawElements(m_nDrawingMode,m_nIndexCnt, GL_UNSIGNED_INT, NULL);
	}
	else
	{
		glDrawArrays(m_nDrawingMode, 0,GetVertexCount());
	}

}
void	CVBOBuffer::RenderArrays(unsigned nCount)
{
	ASSERT(nCount<=GetVertexCount(),"nCount(%d)>VertexCount(%d)",nCount,GetVertexCount());
	glDrawArrays(m_nDrawingMode, 0,nCount);
	
}
void	CVBOBuffer::EndRender()
{

	if(m_sVBODataType & VSD_VERTEX)
	{
		glDisableClientState(GL_VERTEX_ARRAY);
	}
	if(m_sVBODataType & VSD_COLORS)
	{
		glDisableClientState(GL_COLOR_ARRAY);
	}
	//Note Order: We live Client Active State Texture 0
	if(m_sVBODataType & VSD_TEX1)
	{
		glClientActiveTextureARB(GL_TEXTURE1_ARB);
		glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	}
	if((m_sVBODataType & VSD_TEX0) || (m_sVBODataType & VSD_TEX3D))
	{
		glClientActiveTextureARB(GL_TEXTURE0_ARB);
		glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	}
	if(m_sVBODataType & VSD_NORMALS)
	{
		glDisableClientState(GL_NORMAL_ARRAY);
	}
	
	for(LinkedListItem<CShaderAttr*> *pcItem=m_cAttrList.Next(NULL);
		pcItem!=NULL;
		pcItem=m_cAttrList.Next(pcItem))
	{
		pcItem->data->EndUse();
	}

}