#include "Shaders/CShaderObject.hpp"
#include "Globals/Globals.hpp"
#include "Opengl/OpenGLExtensions.h"
#include "Globals/Globals.hpp"
#include "Utils/MiscFuncs.hpp"
//#include <iostream>

#include "ErrorCheck.h"

using namespace std;

bool CShaderObject::m_bShaderOpLocked = false;

CShaderObject::CShaderObject(const char *strName)
{
	m_strName = NULL;

	strAllocNCopy(&m_strName,strName);

	ShaderObject = 0;
	linker_log = 0;
	_mM = false;


	if (!GL_ARB_shader_objects_supported)
		FATALERROR( "**warning** GL_ARB_shader_objects not defined!!\n"); 
	if (!GL_ARB_vertex_shader_supported)
		FATALERROR( "**warning** GL_ARB_vertex_shader not defined!!\n");
	if (!GL_ARB_fragment_shader_supported)
		FATALERROR( "**warning** GL_ARB_fragment_shader not defined!!\n");

	if (!GL_ARB_shading_language_100_supported)
		FATALERROR( "**warning** GL_ARB_shading_language_100 not defined!!\n");
		


	if(g_bGLSL_Supported)
		ShaderObject = glCreateProgramObjectARB();
	
	is_linked = false;       
}

//----------------------------------------------------------------------------- 

CShaderObject::~CShaderObject()
{
	free(m_strName);
	if (linker_log!=0) free(linker_log);
	if (g_bGLSL_Supported)
	{
		for (unsigned int i=0;i<ShaderList.size();i++)
		{
			glDetachObjectARB(ShaderObject, ShaderList[i]->ProgramObject);
			CHECK_GL_ERROR(); // if you get an error here, you deleted the Program object first and then
			// the ShaderObject! Always delete ShaderObjects last!
			if (_mM) delete ShaderList[i]; 
		}                      

		glDeleteObjectARB(ShaderObject);
		CHECK_GL_ERROR();
	}

}

//----------------------------------------------------------------------------- 

bool CShaderObject::oglslEnabled(void)
{
	return g_bGLSL_Supported; 
}

//----------------------------------------------------------------------------- 

void CShaderObject::addShader(CShaderProgram* ShaderProgram)
{
	if (!g_bGLSL_Supported) return;

	if (ShaderProgram==0) return;


	if (!ShaderProgram->is_compiled)
	{
		FATALERROR( "**warning** please compile program before adding object! trying to compile now...\n");
		if (!ShaderProgram->compile())
		{
			FATALERROR( "...compile ERROR!\n");
			return;
		}
		else
		{   
			FATALERROR( "...ok!\n");
		}
	}

	ShaderList.push_back(ShaderProgram); 

}

//----------------------------------------------------------------------------- 

bool CShaderObject::link(void)
{
	if (!g_bGLSL_Supported) return false;

	unsigned int i;

	if (is_linked)  // already linked, detach everything first
	{
		FATALERROR( "**warning** Object is already linked, trying to link again" );
		for (i=0;i<ShaderList.size();i++)
		{
			glDetachObjectARB(ShaderObject, ShaderList[i]->ProgramObject);
			CHECK_GL_ERROR();
		}
	}

	for (i=0;i<ShaderList.size();i++)
	{
		glAttachObjectARB(ShaderObject, ShaderList[i]->ProgramObject);
		CHECK_GL_ERROR();
		//FATALERROR( "attaching ProgramObj [" << i << "] @ 0x" << hex << ShaderList[i]->ProgramObject << " in ShaderObj @ 0x"  << ShaderObject );
	}

	int linked;
	glLinkProgramARB(ShaderObject);
	CHECK_GL_ERROR();
	glGetObjectParameterivARB(ShaderObject, GL_OBJECT_LINK_STATUS_ARB, &linked);
	CHECK_GL_ERROR();

	if (linked)
	{
		is_linked = true;
		return true;
	}
	else
	{
		DEBUGMSG( "**linker error**\n");
	}

	return false;
}

//----------------------------------------------------------------------------- 
// Compiler Log: Ausgabe der Compiler Meldungen in String

const char* CShaderObject::getLinkerLog(void)
{    
	if (!g_bGLSL_Supported) return aGLSLErrorString[0];

	int blen = 0;	
	int slen = 0;	


	if (ShaderObject==0) return aGLSLErrorString[2];

	glGetObjectParameterivARB(ShaderObject, GL_OBJECT_INFO_LOG_LENGTH_ARB , &blen);
	CHECK_GL_ERROR();

	if (blen > 1)
	{
		if (linker_log!=0) 
		{   
			free(linker_log);
			linker_log =0;
		}
		if ((linker_log = (GLcharARB*)malloc(blen)) == NULL) 
		{
			printf("ERROR: Could not allocate compiler_log buffer\n");
			return aGLSLErrorString[3];
		}

		glGetInfoLogARB(ShaderObject, blen, &slen, linker_log);
		CHECK_GL_ERROR();

	}
	if (linker_log!=0)
		return (const char*) linker_log;    

	return aGLSLErrorString[4];
}

void CShaderObject::begin(void)
{
	if (!g_bGLSL_Supported) return;
	if (ShaderObject == 0) return;
	if (m_bShaderOpLocked) return;

	if (is_linked)
	{
		glUseProgramObjectARB(ShaderObject);
		CHECK_GL_ERROR();
	}
}

//----------------------------------------------------------------------------- 

void CShaderObject::end(void)
{
	if (!g_bGLSL_Supported) return;
	if (m_bShaderOpLocked) return;

	glUseProgramObjectARB(0);
	CHECK_GL_ERROR();
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniform1f(const char* varname, GLfloat v0)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform1fARB(loc, v0);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniform2f(const char* varname, GLfloat v0, GLfloat v1)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;
	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform2fARB(loc, v0, v1);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniform3f(const char* varname, GLfloat v0, GLfloat v1, GLfloat v2)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;
	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform3fARB(loc, v0, v1, v2);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniform4f(const char* varname, GLfloat v0, GLfloat v1, GLfloat v2, GLfloat v3)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;
	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform4fARB(loc, v0, v1, v2, v3);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniform1i(const char* varname, GLint v0)
{ 
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform1iARB(loc, v0);

	return true;
}
bool CShaderObject::sendUniform2i(const char* varname, GLint v0, GLint v1)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform2iARB(loc, v0, v1);


	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniform3i(const char* varname, GLint v0, GLint v1, GLint v2)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;
	

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform3iARB(loc, v0, v1, v2);

	return true;
}
bool CShaderObject::sendUniform4i(const char* varname, GLint v0, GLint v1, GLint v2, GLint v3)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform4iARB(loc, v0, v1, v2, v3);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniform1fv(const char* varname, GLsizei count, GLfloat *value)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform1fvARB(loc, count, value);

	return true;
}
bool CShaderObject::sendUniform2fv(const char* varname, GLsizei count, GLfloat *value)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;
	

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform2fvARB(loc, count, value);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniform3fv(const char* varname, GLsizei count, GLfloat *value)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform3fvARB(loc, count, value);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniform4fv(const char* varname, GLsizei count, GLfloat *value)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;
	

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform4fvARB(loc, count, value);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniform1iv(const char* varname, GLsizei count, GLint *value)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;
	

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform1ivARB(loc, count, value);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniform2iv(const char* varname, GLsizei count, GLint *value)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;
	

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform2ivARB(loc, count, value);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniform3iv(const char* varname, GLsizei count, GLint *value)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;
	

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform3ivARB(loc, count, value);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniform4iv(const char* varname, GLsizei count, GLint *value)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;
	

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniform4ivARB(loc, count, value);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniformMatrix2fv(const char* varname, GLsizei count, GLboolean transpose, GLfloat *value)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniformMatrix2fvARB(loc, count, transpose, value);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniformMatrix3fv(const char* varname, GLsizei count, GLboolean transpose, GLfloat *value)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;
	

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniformMatrix3fvARB(loc, count, transpose, value);

	return true;
}

//----------------------------------------------------------------------------- 

bool CShaderObject::sendUniformMatrix4fv(const char* varname, GLsizei count, GLboolean transpose, GLfloat *value)
{
	if (!g_bGLSL_Supported) return false; // GLSL not available
	if (m_bShaderOpLocked) return false;
	

	GLint loc = GetUniLoc(varname);
	if (loc==-1) return false;  // can't find variable

	glUniformMatrix4fvARB(loc, count, transpose, value);

	return true;
}

//----------------------------------------------------------------------------- 

GLint CShaderObject::GetAttribLocation(const GLcharARB *name)
{
	if (m_bShaderOpLocked) return -1;

	GLint loc;

	loc = glGetAttribLocationARB(ShaderObject, name);
	if (loc == -1) 
	{
		FATALERROR( "Error: can't find attribute variable '%s' \n",name);
	}
	CHECK_GL_ERROR();
	return loc;
}
bool CShaderObject::CheckUniformLocation(const GLcharARB *name)
{
	return glGetUniformLocationARB(ShaderObject, name)!=-1;
}
GLint CShaderObject::GetUniLoc(const GLcharARB *name)
{
	if (m_bShaderOpLocked) return -1;
	GLint loc;

	loc = glGetUniformLocationARB(ShaderObject, name);
	if (loc == -1) 
	{
		DEBUGMSG( "Error: can't find uniform variable '%s' \n",name);
	}
	CHECK_GL_ERROR();
	return loc;
}

//----------------------------------------------------------------------------- 

void CShaderObject::GetUniformfv(const char* name, GLfloat* values)
{
	if (!g_bGLSL_Supported) return;
	if (m_bShaderOpLocked) return ;
	GLint loc;

	loc = glGetUniformLocationARB(ShaderObject, name);
	if (loc == -1) 
	{
		FATALERROR( "Error: can't find uniform variable '%s' \n",name);
	}
	glGetUniformfvARB(ShaderObject, loc, values);

}

//----------------------------------------------------------------------------- 

void CShaderObject::GetUniformiv(const char* name, GLint* values)
{
	if (!g_bGLSL_Supported) return;
	if (!g_bGLSL_Supported) return;
	if (m_bShaderOpLocked) return ;


	GLint loc;

	loc = glGetUniformLocationARB(ShaderObject, name);
	if (loc == -1) 
	{
		FATALERROR( "Error: can't find uniform variable '%s' \n",name);
	}

	glGetUniformivARB(ShaderObject, loc, values);

}
