#ifndef CVBOBUFFER_H
#define CVBOBUFFER_H

#include "stdafx.h"
#include "Types/Vector3D.h"
#include "Types/Vector2D.h"
#include "Types/Vector4D.h"
#include "Types/LinkedList.hpp"
#include "CShaderAttribute.hpp"
#include "Opengl/OpenGLExtensions.h"


enum VBO_STOREDATA
{
	VSD_VERTEX  = 1,
	VSD_COLORS  = 2,
	VSD_TEX0    = 4,
	VSD_TEX1    = 8,
	VSD_NORMALS = 16,
	VSD_TEX3D	= 32,
};



class CVBOBuffer 
{
	int					m_sVBODataType;
	unsigned			m_nMaxElements;			//This is our data
	unsigned			m_nMaxIndices;			//Index Count, suppose we use Triangle List, so one tri will consist of 3 indices
	
	bool				m_bUsingIndices;

	GLenum				m_nDrawingMode;			//Triangles or Strips

	GLenum				m_nDataStreamType[6];

	Vector3D			*m_pcVertices;								
	Vector4D			*m_pcColors;
	Vector2D			*m_pcTexCoords0;							
	Vector2D			*m_pcTexCoords1;	
	Vector3D			*m_pcTexCoords3D0;
	Vector3D			*m_pcNormals;		
	unsigned int		*m_pcIndices;	

	

	//ID's to buffers
	unsigned int		m_nVBOVertices;								
	unsigned int		m_nVBOColors;
	unsigned int		m_nVBOTexCoords0;							
	unsigned int		m_nVBOTexCoords1;
	unsigned int		m_nVBOTexCoords3D0;
	unsigned int		m_nVBONormals;	
	unsigned int		m_nVBOIndices;	




	//Count's
	unsigned int		m_nVerticesCnt;								
	unsigned int		m_nColorsCnt;
	unsigned int		m_nTexCoords0Cnt;							
	unsigned int		m_nTexCoords1Cnt;
	unsigned int		m_nTexCoords3D0Cnt;
	unsigned int		m_nNormalsCnt;
	unsigned int		m_nIndexCnt;


	LinkList<CShaderAttr*> m_cAttrList;

private:

	bool ValidateIndices();			//Checks if vertex count is equal to texcoords, color etc
	//Functions
	bool IsVBOSupported();
	bool IsVBOBuilt();


public:
	CVBOBuffer(unsigned _nMaxElements,int _sVBODataType,GLenum _nDrawingMode=GL_TRIANGLES,
			   bool _bUsingIndices=false,unsigned _nIndexCount=0);
	~CVBOBuffer();


	void	SetValuesToZero();
	void	Clear();

	void	AddShaderAttribute(CShaderAttr *pcShaderAttr);
	
	
	//Stream Data
	void		      SetDataStreamType(VBO_STOREDATA _nStoreType,GLenum _eDataType);
	//Pointers
	inline Vector3D * GetVerticesPtr(){return m_pcVertices;}
	inline Vector4D	* GetColorsPtr(){return m_pcColors;}
	inline Vector2D	* GetTexCoords0Ptr(){return m_pcTexCoords0;}
	inline Vector2D	* GetTexCoords1Ptr(){return m_pcTexCoords1;}
	inline Vector3D	* GetTexCoords3D0Ptr(){return m_pcTexCoords3D0;}
	inline Vector3D	* GetNormalsPtr(){return m_pcNormals;}
	inline unsigned int* GetIndexPtr(){return m_pcIndices;}

	unsigned int GetVertexCount(){return m_nVerticesCnt;}		
	unsigned int GetColorCount() {return m_nColorsCnt;}
	unsigned int GetTexCoord0Count(){return	m_nTexCoords0Cnt;}		
	unsigned int GetTexCoord1Count(){return	m_nTexCoords1Cnt;}
	unsigned int GetTexCoord3D0Count(){return m_nTexCoords3D0Cnt;}
	unsigned int GetNormalCount(){return m_nNormalsCnt;	}
	unsigned int GetIndexCnt(){return m_nIndexCnt;	}



	unsigned int GetVertices() {return m_nVBOVertices;}								
	unsigned int GetColors()   {return m_nVBOColors;}	
	unsigned int GetTexCoord0(){return m_nVBOTexCoords0;}								
	unsigned int GetTexCoord1(){return m_nVBOTexCoords1;}	
	unsigned int GetTexCoord3D0(){return m_nVBOTexCoords3D0;}	
	unsigned int GetNormals()  {return m_nVBONormals;}	



	inline   bool tagBuildVBO(GLvoid *pcData,GLuint &nBindID,GLuint nDataCount,GLuint nSize,GLenum nDataType)
	{
		if(pcData!=NULL && nBindID==0 && nDataCount>0)
		{
			glGenBuffersARB( 1, &nBindID );						
			glBindBufferARB( GL_ARRAY_BUFFER_ARB, nBindID );			
			glBufferDataARB( GL_ARRAY_BUFFER_ARB, nDataCount*nSize, pcData, nDataType );
			return true;
		}
		else return false;
	}
	inline bool tagUpdateVBO(GLvoid *pcData,const GLuint &nBindID,GLuint nDataCount,GLuint nSize)
	{
		if(pcData!=NULL && nBindID!=0 && nDataCount>0)
		{
			glBindBufferARB( GL_ARRAY_BUFFER_ARB, nBindID );			
			glBufferDataARB( GL_ARRAY_BUFFER_ARB, nDataCount*nSize, pcData, GL_DYNAMIC_DRAW_ARB );
			return true;
		}
		else return false;
	}
public:
	//Index
	void	AddIndex(unsigned _nIndex);
	//Vertex
	void	AddVertex(Vector3D v);
	void	AddVertex(float x,float y,float z)
	{
			AddVertex(Vector3D(x,y,z));
	}
	//Color
	void	AddColor(Vector4D c);
	void	AddColor(float r,float g,float b,float a)
	{
		AddColor(Vector4D(r,g,b,a));
	}
	//TexCoord0
	void	AddTexCoord0(Vector2D t);
	void	AddTexCoord0(float u,float v)
	{
		AddTexCoord0(Vector2D(u,v));
	}
	//TexCoord1
	void	AddTexCoord1(Vector2D t);
	void	AddTexCoord1(float u,float v)
	{
		AddTexCoord1(Vector2D(u,v));
	}

	//Normals
	void	AddNormal(Vector3D n);
	void	AddNormal(float x,float y,float z)
	{
		AddNormal(Vector3D(x,y,z));
	}

	//TexCoord3D0
	void	AddTexCoord3D0(Vector3D t);
	void	AddTexCoord3D0(float u,float v,float w)
	{
		AddTexCoord3D0(Vector3D(u,v,w));
	}
	
	//For Dynamic Buffers only
	void	UpdateVertexBuffer();
	void	UpdateNormalsBuffer();
	void	UpdateColorsBuffer();



	void	BuildVBO();



	//Render Functions

	void	BeginRender();
	void	Render();
	void	RenderArrays(unsigned nCount);
	void	EndRender();

};



#endif