#ifndef LOOKUPTABLE_HPP
#define LOOKUPTABLE_HPP

#include <windows.h>
#include "Types/Math1D.h"
#include "Types/Vector3D.h"

#include "ErrorCheck.h"

const static unsigned g_nLookupPreTableDetail=10;

template <class ItemData>class LookupTable
{
private:
    int		  m_nMaxPoints;

    ItemData *m_pcPoints;                               // First point at x = xMin, last at x = xMax

    float	  m_nMin, m_nMax,m_nSize;					// Interpolating value

    float	  m_nScale;									// 

	ItemData  m_vCatmullPoints[4];

	bool	  m_bPreCreateTable;



	//For Catmull Rom Calculation
	void	 CalculateCatmullPoints(
		const ItemData& p0,const ItemData&  p1,const ItemData&  p2,const ItemData&  p3);

	inline float LT_Length(const float& v1,const float& v2)
	{
		return fabsf(v2-v1);
	}

	inline float LT_Length(const Vector3D& v1,const Vector3D& v2)
	{
		return Vector3D(v2-v1).length();
	}
private:
	LookupTable <ItemData> *m_pcHerpTable;
	LookupTable <ItemData> *m_pcBHerpTable;
	LookupTable <ItemData> *m_pcCHerpTable;
public: 
	
    LookupTable();
	~LookupTable() { FreeTable(); }

	//Misc
	int  GetPointCount()
	{
		return m_nMaxPoints;
	}
	float GetMinX()
	{
		return m_nMin;
	}
	float GetMaxX()
	{
		return m_nMax;
	}
	ItemData GetPoint(int _nIndex)
	{
		if(_nIndex>-1 && _nIndex<m_nMaxPoints)
			return m_pcPoints[_nIndex];
		else
		{
			ItemData dummy;
			FATALERROR("Out of Range [0;%d] , got %d",m_nMaxPoints,_nIndex);
			return dummy;
		}
	}



	void  CreateTable(int _nMaxPoints, ItemData *_pcPoints, float _nMin, float _nMax,bool _bPreCreateTable=false);

    void  FreeTable();

    ItemData Lerp( const float& x);

    ItemData Herp( const float& x);

    ItemData BezierHerp(const float& x);


    ItemData CatmullHerp(const float& x);


	//Loading/Saving
	void	SaveLookupTable(const char *strFName);
	void	LoadLookupTable(const char *strFName);


    
};







/******************************************************************************
*   LookupTable
*
*   Constructor.
******************************************************************************/

template <class ItemData>
LookupTable<ItemData>::LookupTable()
:m_nMaxPoints(0)
,m_pcPoints(NULL)
,m_bPreCreateTable(false)
,m_pcHerpTable(NULL)
,m_pcBHerpTable(NULL)
,m_pcCHerpTable(NULL)

{
	ASSERT(g_nLookupPreTableDetail>0,"Incorrect g_nLookupPreTableDetail=%d",g_nLookupPreTableDetail);
}

/******************************************************************************
*   FreeTable
*
*   FreeTable the memory.
******************************************************************************/

template <class ItemData>
void LookupTable<ItemData>::FreeTable()
{
	m_nMaxPoints = 0;
	m_bPreCreateTable=false;
    if (m_pcPoints) {
        delete [] m_pcPoints;
        m_pcPoints = NULL;
    }
	if(m_pcHerpTable)
	{
		delete m_pcHerpTable;
		m_pcHerpTable=NULL;
	}
	if(m_pcBHerpTable)
	{
		delete m_pcBHerpTable;
		m_pcBHerpTable=NULL;
	}
	if(m_pcCHerpTable)
	{
		delete m_pcCHerpTable;
		m_pcCHerpTable=NULL;
	}

}



template <class ItemData>
void LookupTable<ItemData>::CreateTable(int _nMaxPoints, ItemData *_pcPoints, float _nMin, float _nMax,bool _bPreCreateTable)
{
	

    FreeTable();

    ASSERT(_nMaxPoints >= 3,"_nMaxPoints must be at least 3 :%d",_nMaxPoints);
	ASSERT(_nMax > _nMin,"nMax<nMin");

    m_nMaxPoints = _nMaxPoints;

    m_pcPoints = new ItemData[_nMaxPoints];

    memcpy(m_pcPoints, _pcPoints, sizeof(ItemData) * _nMaxPoints);
 
    m_nMin = _nMin;

    m_nMax = _nMax-EPS;

	m_nSize=_nMax-_nMin;

    m_nScale = (_nMaxPoints - 1) / (_nMax - _nMin);

	if(_bPreCreateTable)
	{
		//Create Herp Table
		int m_nPreMaxPoints=m_nMaxPoints*g_nLookupPreTableDetail;
		ItemData *pcTmpPoints=new ItemData[m_nPreMaxPoints];
		float nStep=(_nMax-_nMin)/float(m_nPreMaxPoints-1);
		float nIn=_nMin;
		int   i;
		for(i=0;i<m_nPreMaxPoints;i++)
		{
			pcTmpPoints[i]=Herp(nIn);
			nIn+=nStep;
		}
		m_pcHerpTable=new LookupTable<ItemData>;
		m_pcHerpTable->CreateTable(m_nPreMaxPoints,pcTmpPoints,_nMin,_nMax);
		
		//Create BHerp Table
		nIn=_nMin;
		for(i=0;i<m_nPreMaxPoints;i++)
		{
			pcTmpPoints[i]=BezierHerp(nIn);
			nIn+=nStep;
		}
		m_pcBHerpTable=new LookupTable<ItemData>;
		m_pcBHerpTable->CreateTable(m_nPreMaxPoints,pcTmpPoints,_nMin,_nMax);
		//Create CHerp Table
		if(m_nMaxPoints==4)
		{
			nIn=_nMin;
			for(i=0;i<m_nPreMaxPoints;i++)
			{
				pcTmpPoints[i]=CatmullHerp(nIn);
				nIn+=nStep;
			}
			m_pcCHerpTable=new LookupTable<ItemData>;
			m_pcCHerpTable->CreateTable(m_nPreMaxPoints,pcTmpPoints,_nMin,_nMax);
		}

		delete []pcTmpPoints;
	}
	m_bPreCreateTable=_bPreCreateTable;

}


/******************************************************************************
*   Lerp
*
*   params :
*       float x                             -   input value (between nMin and nMax)
*   returns :
*       float                               -   y = f(x)
*
*   Lookup y = f(x). Uses linear interpolation.
******************************************************************************/

template <class ItemData>
ItemData LookupTable<ItemData>::Lerp(const float& x)
{
    ASSERT(m_pcPoints,"m_pcPoints==NULL");
	//ASSERT(x>m_nMin-EPS && x<m_nMax+EPS,"x is not in range");
	
	//X must be in range
    if (x < m_nMin)return m_pcPoints[0];    
	if (x > m_nMax)return m_pcPoints[m_nMaxPoints - 1];   
	

    float i = (x - m_nMin) * m_nScale;

    int index = int(i);

    ASSERT(index > -1 && index < (m_nMaxPoints - 1),"Index was calculated incorrectly ");
	// value of x is between p1 and p2
    ItemData p1 = m_pcPoints[index];      

    ItemData p2 = m_pcPoints[index + 1];

    float t = i - index;                            // 0.0 <= t < 1.0

    ASSERT(t >= 0.0 && t < 1.0,"t range must be 0.0f..1.0f");

    return p1 + t * (p2 - p1);
}

/******************************************************************************
*   herp
*
*   params :
*       float x                             -   input value (between xMin and xMax)
*   returns :
*       float                               -   y = f(x)
*
*   Lookup y = f(x). Uses Hermite interpolation.
*
*   Hermite interpolate between p1 and p2. p0 and p3 are used for finding gradient at p1 and p2.
*       value = p0 * (2t^2 - t^3 - t)/2
*            + p1 * (3t^3 - 5t^2 + 2)/2
*            + p2 * (4t^2 - 3t^3 + t)/2
*            + p3 * (t^3 - t^2)/2
******************************************************************************/

template <class ItemData>
ItemData LookupTable<ItemData>::Herp(const float& x)
{
    ASSERT(m_pcPoints,"m_pcPoints==NULL");
	//ASSERT(x>m_nMin-EPS && x<m_nMax+EPS,"x is not in range");
	
	//X must be in range
    if (x < m_nMin)return m_pcPoints[0];    
	if (x > m_nMax)return m_pcPoints[m_nMaxPoints - 1];    
	
	if(m_bPreCreateTable)
	{
		return m_pcHerpTable->Lerp(x);
	}
	else
	{
		float i = (x - m_nMin) * m_nScale;                // 0.0 <= i <= (numPoints - 2)

		int index = int(i);

		ASSERT(index > -1 && index < (m_nMaxPoints - 1),"Index was calculated incorrectly ");


		ItemData p1 = m_pcPoints[index];						// value of x is between p1 and p2
		ItemData p2 = m_pcPoints[index + 1];
		ItemData p0, p3;										// p0 and p3 are on outside of p1 and p2
		float t = i - index;									// 0.0 <= t < 1.0
		float t2 = t * t;
		float t3 = t2 * t;
		float kp0 = 2.0f * t2 - t3 - t ;						// hermite blending factors
		float kp1 = 3.0f * t3 - 5.0f * t2 + 2.0f ;
		float kp2 = 4.0f * t2 - 3.0f * t3 + t ;
		float kp3 = t3 - t2 ;

		if (index == 0)											// special case where p0 is off the end
			p0 = 2.0f * p1 - p2;
		else
			p0 = m_pcPoints[index - 1];
		if (index == m_nMaxPoints - 2)							// special case where p3 is off the end
			p3 = 2.0f * p2 - p1;
		else
			p3 = m_pcPoints[index + 2];
		return (p0 * kp0 + p1 * kp1 + p2 * kp2 + p3 * kp3)/2.0f;
	}
}

template <class ItemData>
ItemData LookupTable<ItemData>::BezierHerp(const float& x)
{
	ASSERT(m_nMaxPoints<21,"Performance warning Bezier use function Factorial, and number will be generated too big,%d is too big",m_nMaxPoints);
	//X must be in range
    if (x < m_nMin)return m_pcPoints[0];    
	if (x > m_nMax)return m_pcPoints[m_nMaxPoints - 1];    

	if(m_bPreCreateTable)
	{
		return m_pcBHerpTable->Lerp(x);
	}
	else
	{
		float t=(x-m_nMin)/m_nSize;
		ItemData sum=m_pcPoints[0]*(float)Bernstein(m_nMaxPoints-1,0,t);

		for(int j=1;j<m_nMaxPoints;j++)
		{
			sum=sum+m_pcPoints[j]*(float)Bernstein(m_nMaxPoints-1,j,t);
		}
		return sum;
	}

  
}

template <class ItemData>
void	 LookupTable<ItemData>::CalculateCatmullPoints(const ItemData& p0,const ItemData&  p1,const ItemData&  p2,const ItemData&  p3)
{

	float delta[3];

	delta[0]=LT_Length(p0,p1);if(delta[0]<EPS)delta[0]=EPS;
	delta[1]=LT_Length(p1,p2);if(delta[1]<EPS)delta[1]=EPS;
	delta[2]=LT_Length(p2,p3);if(delta[2]<EPS)delta[2]=EPS;

    
	
	ItemData l[2];
	l[0]=(delta[1]-delta[0])/(delta[1]*delta[0])*p1 +
		delta[0]/(delta[1]*(delta[0]+delta[1]))*p2-
		delta[1]/(delta[0]*(delta[0]+delta[1]))*p0;
	l[1]=(delta[2]-delta[1])/(delta[2]*delta[1])*p2 +
		delta[1]/(delta[2]*(delta[1]+delta[2]))*p3-
		delta[2]/(delta[1]*(delta[1]+delta[2]))*p1;




    m_vCatmullPoints[0]=p1;
    m_vCatmullPoints[1]=p1+(delta[1]/3.0f)*l[0];
    m_vCatmullPoints[2]=p2-(delta[1]/3.0f)*l[1];
    m_vCatmullPoints[3]=p2;

}
template <class ItemData>
ItemData LookupTable<ItemData>::CatmullHerp(const float& x)
{
	ASSERT(m_nMaxPoints==4,"Works with four points only");

	if(m_bPreCreateTable)
	{
		return m_pcCHerpTable->Lerp(x);
	}
	else
	{
		ItemData p0= m_pcPoints[0];						// value of x is between p1 and p2
		ItemData p1= m_pcPoints[1];						// value of x is between p1 and p2
		ItemData p2= m_pcPoints[2];						// value of x is between p1 and p2
		ItemData p3= m_pcPoints[3];						// value of x is between p1 and p2
		
		
		CalculateCatmullPoints(p0,p1,p2,p3);

		float t=(x-m_nMin)/m_nSize;
		t=fClamp(t,0.0f,1.0f);


		float OneMinusT=(1-t);
		float OneMinusTSq=OneMinusT*OneMinusT;
		float TSq=t*t;

		return (OneMinusT*OneMinusTSq*m_vCatmullPoints[0]+3*OneMinusTSq*t*m_vCatmullPoints[1]+3*OneMinusT*TSq*m_vCatmullPoints[2]+t*TSq*m_vCatmullPoints[3]);
	}
}

template <class ItemData>
void	LookupTable<ItemData>::SaveLookupTable(const char *strFName)
{
#ifdef USE_DIRECTX
	int sz=sizeof(ItemData);
	int i;
	FILE *fp=fopen(strFName,"wb");
	if(fp==NULL)FATALERROR1("Couldnt open %s for saving",strFName);
	else
	{
		fwrite(&sz,sizeof(int),1,fp);
		fwrite(&m_nMaxPoints,sizeof(int),1,fp);
		fwrite(&m_nMin,sizeof(float),1,fp);
		fwrite(&m_nMax,sizeof(float),1,fp);
		fwrite(&m_bPreCreateTable,sizeof(bool),1,fp);
		for(i=0;i<m_nMaxPoints;i++)
		{
			fwrite(&(m_pcPoints[i]),sizeof(ItemData),1,fp);
		}
		fclose(fp);
	}
#endif
#ifdef USE_PS2
	FATALERROR0("Not Implemented on PS2");
#endif

}

template <class ItemData>
void	LookupTable<ItemData>::LoadLookupTable(const char *strFName)
{
#ifdef USE_DIRECTX
	int sz=sizeof(ItemData),fz;
	int i;
	int		_nMaxPoints;
	float	_nMin;
	float   _nMax;
	bool	_bPreCreate;
	ItemData *_pcPoints;

	FILE *fp=fopen(strFName,"rb");
	if(fp==NULL)FATALERROR1("Couldnt open %s for saving",strFName);
	else
	{
		fread(&fz,sizeof(int),1,fp);
		if(fz!=sz)
		{
			fclose(fp);
			FATALERROR2("File Item Size is %d and current's LookupTable size is %d",fz,sz);
		}


		fread(&_nMaxPoints,sizeof(float),1,fp);
		fread(&_nMin,sizeof(int),1,fp);
		fread(&_nMax,sizeof(float),1,fp);
		fread(&_bPreCreate,sizeof(bool),1,fp);

		_pcPoints=new ItemData[_nMaxPoints];
		for(i=0;i<_nMaxPoints;i++)
		{
			fread(&(_pcPoints[i]),sizeof(ItemData),1,fp);
		}
		fclose(fp);
		CreateTable(_nMaxPoints,_pcPoints,_nMin,_nMax,_bPreCreate);

		delete []_pcPoints;
	}
#endif
#ifdef USE_PS2
	FATALERROR0("Not Implemented on PS2");
#endif

}

#include "NoMemoryMan.h"

#endif

