﻿using System;
using System.Collections.Generic;
using System.Text;
using Elf2D;
using Elf2D.Render;
using Elf2D.Input;
using Elf2D.Maths;
using TheStory.Effects;

namespace TheStory {
	class Game {
		public const int WindowWidth = 800;
		public const int WindowHeight = 600;

		public const float NextMessageTime = 1.0f;
		public const float NextMessageTimeIncreaseSpeed = 0.01f;

		public const float NextBonusTime = 10.0f;

		Random random = new Random();

		LoveIndicator loveIndicator;

		Player malePlayer;
		Player femalePlayer;

		List<Effect> effects = new List<Effect>();
		List<Message> messages = new List<Message>(32);
		float nextMessageTimer = 0.0f;
		PlayerGender nextThrower = PlayerGender.Female;

		float gameTime = 0;

		bool matrixEffectOn;
		float bonusTimer = 2.0f;

		Font font;


		internal LoveIndicator LoveIndicator {
			get { return loveIndicator; }
		}

		internal bool HasEnded { get; private set; }


		internal void AddEffect(Explosion explosion) {
			effects.Add(explosion);
		}

		internal void Update(float deltaTime) {
			//System.Console.WriteLine(deltaTime);
			if (deltaTime > 1.0f / 60.0f) {
				deltaTime = 1.0f / 60.0f;
			}
			gameTime += deltaTime;

			if (Keyboard.IsKeyDown(Key.Space)) {
				matrixEffectOn = true;
			} else {
				matrixEffectOn = false;
			}

			if (matrixEffectOn) {
				deltaTime *= 0.3f;
				malePlayer.ChangeLove(-deltaTime * 10.0f);
				femalePlayer.ChangeLove(-deltaTime * 10.0f);
			}

			GameTime time = new GameTime(deltaTime, gameTime);

			UpdateBonus(time);
			UpdateMessages(time);
			UpdateEffects(time);
			CheckCollisions();

			malePlayer.Update(time);
			femalePlayer.Update(time);

			loveIndicator.SetFemale(femalePlayer.Love);
			loveIndicator.SetMale(malePlayer.Love);
			loveIndicator.Update(time);

			if (femalePlayer.Love <= 0.0f || malePlayer.Love <= 0.0f) {
				HasEnded = true;
			}

			if (Keyboard.IsKeyPressed(Key.Escape)) {
				System.Diagnostics.Process.GetCurrentProcess().CloseMainWindow();
			}
		}

		private void UpdateEffects(GameTime time) {
			foreach (Effect effect in effects) {
				effect.Update(time);
			}

		RESTART:
			foreach (Effect effect in effects) {
				if (effect.IsDead) {
					effects.Remove(effect);
					goto RESTART;
				}
			}
		}

		private void UpdateBonus(GameTime time) {
			bonusTimer -= time.DeltaTime;
			if (bonusTimer < 0.0f) {
				bonusTimer = NextBonusTime;
				SpawnNewBonus();
			}
		}

		private void SpawnNewBonus() {
			Player catcher;
			PlayerGender catcherGender;

			switch (nextThrower) {
				case PlayerGender.Female:
					catcher = femalePlayer;
					catcherGender = PlayerGender.Female;
					break;

				case PlayerGender.Male:
					catcher = malePlayer;
					catcherGender = PlayerGender.Male;
					break;

				default:
					throw new Exception();
			}

			Message bonus = new Message(catcherGender, GenerateBonusType(), catcher.Position, 0.0f, 0.0f);
			messages.Add(bonus);
		}

		private void UpdateMessages(GameTime time) {
			nextMessageTimer -= time.DeltaTime;
			if (nextMessageTimer < 0.0f) {
				nextMessageTimer = NextMessageTime - time.TotalGameTime * NextMessageTimeIncreaseSpeed;
				if (nextMessageTimer < 0.1f) {
					nextMessageTimer = 0.1f;
				}
				ThrowNextMessage();
			}

			foreach (Message message in messages) {
				message.Update(time);
			}

		RESTART:
			foreach (Message message in messages) {
				if (message.IsDead) {
					messages.Remove(message);
					goto RESTART;
				}
			}
		}

		private void CheckCollisions() {
			foreach (Message message in messages) {
				Collide(malePlayer, message);
				Collide(femalePlayer, message);
			}
		}

		private void Collide(Player player, Message message) {
			if (player.Gender != message.Catcher) {
				return;
			}

			float distance = Vector.Distance(player.Position, message.Position);
			if (distance + 10.0f < Player.PlayerRadius + Message.MessageRadius) {
				player.OnCollision(this, message);
				message.OnCollision(player);
			}		
		}

		private void ThrowNextMessage() {
			Player thrower = null;
			PlayerGender catcherGender;

			switch (nextThrower) {
				case PlayerGender.Female:
					thrower = femalePlayer;
					catcherGender = PlayerGender.Male;
					break;

				case PlayerGender.Male:
					thrower = malePlayer;
					catcherGender = PlayerGender.Female;
					break;

				default:
					throw new Exception();
			}

			nextThrower = catcherGender;

			Message message = new Message(catcherGender, GenerateMessageType(), thrower.Position, thrower.GetThrowDirection(), 100.0f);
			messages.Add(message);
		}

		private MessageType GenerateMessageType() {
			int msg = random.Next((int)MessageType._LastFlyingMessage + 1);
			return (MessageType)msg;
		}

		private MessageType GenerateBonusType() {
			int msg = random.Next((int)MessageType._FirstBonus, (int)MessageType._LastBonus + 1);
			return (MessageType)msg;
		}

		internal void Render() {
			Video.Clear(0xFFFFFFFF);
			Video.EnableBlending(true);

			foreach (Effect effect in effects) {
				effect.Render();
			}

			foreach (Message message in messages) {
				message.Render();
			}

			malePlayer.Render();
			femalePlayer.Render();

			loveIndicator.Render();
		}

		internal void Init() {
			malePlayer = new Player(PlayerGender.Male);
			femalePlayer = new Player(PlayerGender.Female);

			loveIndicator = new LoveIndicator();
		}

		internal void Reset() {
			femalePlayer.Reset();
			malePlayer.Reset();

			effects.Clear();
			messages.Clear();
			nextMessageTimer = 0.0f;
			nextThrower = PlayerGender.Female;
			gameTime = 0;
			matrixEffectOn = false;
			bonusTimer = 2.0f;

			HasEnded = false;
		}
	}
}
