﻿using System;
using System.Collections.Generic;
using System.Text;
using Elf2D.Render;
using Elf2D.Maths;
using Elf2D;
using Elf2D.Audio;

namespace TheStory {
	enum MessageType {
		Heart,
		SweetTalk,
		Present,

		BadHeart,
		BadTalk,
		WWWWWWWWWWWWW,

		_LastFlyingMessage = WWWWWWWWWWWWW,

		Phone,
		Tv,
		Broom,

		_FirstBonus = Phone,
		_LastBonus = Broom,

		_NumMessageTypes
	}


	class Message {
		internal const float MessageSize = 50.0f;
		internal const float MessageRadius = MessageSize / 2.0f;
		internal const float AppearTime = 5.0f;
		internal const float AppearTimeDelay = 3.0f;

		private static string[] TextureName = 
		{
			@"Data\Textures\MessageHeart.tga",
			@"Data\Message2.tga",
			@"Data\Textures\MessagePresent.tga",
			@"Data\MessageBad1.tga",
			@"Data\MessageBad2.tga",
			@"Data\MessageBad3.tga",
			@"Data\BonusPhone.tga",
			@"Data\BonusTv.tga",
			@"Data\BonusBroom.tga",
		};

		Texture texture;
		Vector position;
		Vector velocity;
		float appearTimer;
		float lifeTime;
		Sound appearSound;


		internal MessageType MessageType { get; private set; }
		internal PlayerGender Catcher { get; private set; }
		internal bool IsDead { get; private set; }

		internal Vector Position {
			get { return position; }
			set { position = value; }
		}

		internal void OnCollision(Player player) {
			IsDead = true;
		}

		internal void Update(GameTime time) {
			lifeTime += time.DeltaTime;

			appearTimer -= time.DeltaTime;
			if (appearTimer < 0.0f) {
				appearTimer = 0.0f;
			}

			position += velocity * time.DeltaTime;

			if (position.x < 00.0f) {
				IsDead = true;
			} else if (position.x > (float)Game.WindowWidth + 0.0f) {
				IsDead = true;
			}
		}

		internal void Render() {
			texture.Bind();

			float alpha;
			if (appearTimer < AppearTimeDelay) {
				alpha = 1.0f - (appearTimer / (AppearTime - AppearTimeDelay));
			} else {
				alpha = 0.15f;
			}
			alpha = MathUtils.Clamp(alpha, 0.15f, 1.0f);

			float size = MessageSize;
			if (lifeTime > 0.0f && lifeTime < 0.3f) {
				float t = (float)lifeTime / 0.3f;
				size = MessageSize * t;
			} else if (lifeTime > 0.3f && lifeTime < 0.5f) {
				float t = (lifeTime - 0.3f) / 0.2f;
				size = MessageSize + 3.0f * (float)Math.Sin(t * Math.PI * 4.0f) * (1.0f - t);
			}

			if (MessageType == MessageType.Phone || MessageType == MessageType.Broom || MessageType == MessageType.Tv) {
				alpha = 1.0f;
			}

			Quads.Begin();
			Quads.SetColour(0x00FFFFFF | (uint)(alpha * 255.0f) << 24);
			Quads.SetTextureRectangle(0, 0, 1, 1);
			Quads.DrawRectangleCentered(position.x, position.y, size, size);
			Quads.End();
		}

		internal Message(PlayerGender catcherGender, MessageType messageType, Vector position, float throwDirection, float throwSpeed) {
			this.appearTimer = AppearTime;
			this.Catcher = catcherGender;
			this.MessageType = messageType;

			int type = (int)messageType;
			texture = Texture.Load(TextureName[type], TextureName[type]);

			bool isBonus = messageType == MessageType.Phone || messageType == MessageType.Tv || messageType == MessageType.Broom;

			if (isBonus) {
				this.position = position;
				Random rand = new Random();
				do {
					this.position.y = (float)rand.NextDouble() * (Player.PlayerBorderUpper - Player.PlayerBorderLower) + Player.PlayerBorderLower;
				} while (Vector.SquareDistance(this.position, position) < 200.0f * 200.0f);
			} else {
				this.position = position + Vector.XAxis * throwDirection * 30.0f;
			}

			this.velocity = new Vector(throwDirection, 0.0f);

			if (!isBonus) {
				Vector targetPoint = new Vector(position.x + throwDirection * 1000.0f, (float)(new Random()).NextDouble() * (Player.PlayerBorderUpper - Player.PlayerBorderLower) + Player.PlayerBorderLower);
				velocity = targetPoint - position;
			}

			this.velocity.Normalize();
			this.velocity.Length *= throwSpeed;

			appearSound = Sound.Load("Message", @"Data\Sounds\Message.wav");
			appearSound.Play(0.9f);
		}
	}
}
